import torch as t
import argparse
import pandas as pd
from pathlib import Path
from src.probing.utils import load_activations_and_labels, save_results
from src.probing.metrics import calculate_auroc


SPLIT_RATIO = 0.8

class LRProbe(t.nn.Module):
    def __init__(self, d_in):
        super().__init__()
        self.net = t.nn.Sequential(
            t.nn.Linear(d_in, 1, bias=False, dtype=t.bfloat16),
        )

    def forward(self, x):
        return self.net(x).squeeze()  # Remove all singleton dimensions

    def normalize(self, activations):
        return activations - activations.mean(dim=0, keepdim=True)
    
    def train_probe(self, activations, labels, lr=0.001, weight_decay=0.1, epochs=1000):
        activations = self.normalize(activations)
        print("Shape after normalize:", activations.shape)
        
        opt = t.optim.AdamW(self.net.parameters(), lr=lr, weight_decay=weight_decay)
        for _ in range(epochs):
            opt.zero_grad()
            loss = t.nn.BCEWithLogitsLoss()(self(activations), labels)
            loss.backward()
            opt.step()
        

def main():
    parser = argparse.ArgumentParser(description="Train linear probes on model activations")
    parser.add_argument("--activations_dir", default="activations",
                        help="Directory containing saved activations and labels")
    parser.add_argument("--layer", type=int, required=True,
                        help="Layer number to probe")
    parser.add_argument("--device", default="cuda" if t.cuda.is_available() else "cpu",
                        help="Device to run training on")
    parser.add_argument("--model", default="meta-llama/Llama-3.1-8B",
                        help="Model name")
    parser.add_argument("--probes_dir", default="results/probes",
                        help="Directory to save probes")
    parser.add_argument("--token", default="last", choices=["last", "max", "mean"], 
                        help="Token(s) to probe")
    # binary
    parser.add_argument("--contrast", default=True, action="store_true",
                        help="Use contrast labels")
    args = parser.parse_args()

    device = args.device  # Use the device from args consistently

    processed_activations = []
    # TODO: for GoT
    if args.contrast:
        activations_false, labels = load_activations_and_labels(args.activations_dir / "contrast_False", args.layer, device=device)
        activations_true, _ = load_activations_and_labels(args.activations_dir / "contrast_True", args.layer, device=device)

        for act_false, act_true in zip(activations_false, activations_true):
            act = act_false - act_true
            if args.token == "last":
                act = act[:, -1, :].squeeze()  
            processed_activations.append(act)
    else:
        activations, labels = load_activations_and_labels(args.activations_dir / "No_Contrast", args.layer, device=device)
        for act in activations:
            print("Shape before selecting last token:", act.shape)
            if args.token == "last":
                act = act[:, -1, :].squeeze()  
            processed_activations.append(act)
    
    activations = t.stack(processed_activations, dim=0)
    print("Shape after stacking:", activations.shape)

    random_idx = t.randperm(len(activations))
    activations = activations[random_idx]
    labels = labels[random_idx]

    split_idx = int(len(activations) * SPLIT_RATIO)
    train_activations, train_labels = activations[:split_idx], labels[:split_idx]
    test_activations, test_labels = activations[split_idx:], labels[split_idx:]

    # Move to device
    train_activations = train_activations.to(device)
    train_labels = train_labels.to(device)
    test_activations = test_activations.to(device)
    test_labels = test_labels.to(device)

    # Train the probe
    probe = LRProbe(train_activations.shape[-1]).to(device)  # Move probe to device
    probe.train_probe(train_activations, train_labels)

    print("Evaluating on test set")
    with t.no_grad():
        test_scores = probe(test_activations)
    test_auroc = calculate_auroc(test_scores, test_labels, device)
    print(f"Test AUROC: {test_auroc}")

    probes_dir = Path(args.probes_dir) / args.model / args.activations_dir.split('/')[-1]
    probes_dir.mkdir(parents=True, exist_ok=True)

    # Save the trained probe weights
    probe_save_path = probes_dir / f"probe_layer_{args.layer}.pt"
    t.save({
        'state_dict': probe.state_dict(), # wheights only
        'config': {
            'input_dim': train_activations.shape[-1],
            'layer': args.layer
        }
    }, probe_save_path)
    print(f"Saved trained probe to {probe_save_path}")

    # Save results
    output_dir = probes_dir / "train_results.csv"
    save_results(
        scores=test_scores,
        labels=test_labels,
        auroc=test_auroc,
        model_name=args.model,
        layer=args.layer,
        dataset_name=args.activations_dir.split('/')[-1],
        output_dir=output_dir,
        token=args.token,
        include_prompt=args.include_prompt
    )

if __name__ == "__main__":
    main()
